from rest_framework import status
from rest_framework.decorators import api_view
from rest_framework.generics import get_object_or_404, RetrieveAPIView
from rest_framework.response import Response
from rest_framework.views import APIView
from rest_framework.permissions import IsAuthenticated
from settings.models import NumberingTemplate
from .models import Payment
from .serializers import InvoiceSerializer
from django.db import transaction, models
from django.utils import timezone
import os
from django.core.files.base import ContentFile


class InvoiceView(RetrieveAPIView):
    queryset = Payment.objects.all()
    serializer_class = InvoiceSerializer
    lookup_field = 'pk'
    permission_classes = [IsAuthenticated]

    def get_queryset(self):
        return Payment.objects.filter(user=self.request.user)


class InvoiceAPIView(APIView):
    permission_classes = [IsAuthenticated]

    def post(self, request):
        data = request.data.copy()
        user = request.user

        # Получаем ID выбранного шаблона
        template_id = int(data.get('template'))
        if not template_id:
            return Response(
                {"detail": "Template ID is required"},
                status=status.HTTP_400_BAD_REQUEST
            )
        if template_id not in [1, 2, 3, 4]:

            try:

                template = NumberingTemplate.objects.get(
                    id=template_id,
                    user=user
                )
            except NumberingTemplate.DoesNotExist:
                return Response(
                    {"detail": "Template not found"},
                    status=status.HTTP_404_NOT_FOUND
                )
        else:
            template = NumberingTemplate.objects.get(
                id=template_id,
            )
        # Генерируем номер счета
        invoice_number = template.generate_number()
        data['invoice_number'] = invoice_number
        data['user'] = user.id
        data['template'] = template.pk

        serializer = InvoiceSerializer(data=data)
        serializer.is_valid(raise_exception=True)

        with transaction.atomic():
            if template_id in [1, 2, 3, 4]:
                for i in NumberingTemplate.objects.filter(id__in=[1, 2, 3, 4]):
                    i.next_number += 1
                    i.save()

            payment = serializer.save()
            # Увеличиваем счетчик в шаблоне
            template.next_number += 1
            template.save()

        return Response(
            InvoiceSerializer(payment).data,
            status=status.HTTP_201_CREATED
        )

    def put(self, request, pk=None):
        if not pk:
            return Response(status=status.HTTP_405_METHOD_NOT_ALLOWED)

        instance = get_object_or_404(Payment, id=pk, user=request.user)

        # Удаляем поле template из данных для обновления
        data = request.data.copy()
        data.pop('template', None)

        serializer = InvoiceSerializer(
            instance=instance,
            data=data,
            partial=True
        )
        serializer.is_valid(raise_exception=True)
        updated_payment = serializer.save()

        return Response(
            InvoiceSerializer(updated_payment).data,
            status=status.HTTP_200_OK
        )

    def delete(self, request, pk=None):
        if not pk:
            return Response(status=status.HTTP_405_METHOD_NOT_ALLOWED)

        instance = get_object_or_404(Payment, id=pk, user=request.user)
        instance.delete()
        return Response(
            {"message": "Payment deleted successfully"},
            status=status.HTTP_200_OK
        )


@api_view(http_method_names=['POST'])
def duplicate(request, pk):
    try:
        original_payment = Payment.objects.get(pk=pk, user=request.user)
    except Payment.DoesNotExist:
        return Response(
            {"detail": "Payment not found or you don't have permission"},
            status=status.HTTP_404_NOT_FOUND
        )

    try:
        with transaction.atomic():
            # Генерируем новый номер счета если есть шаблон
            if original_payment.template:
                invoice_number = original_payment.template.generate_number()
                original_payment.template.next_number += 1
                original_payment.template.save()
            else:
                invoice_number = f"{original_payment.invoice_number} (Копия)"

            # Создаем новый платеж
            new_payment = Payment.objects.create(
                user=request.user,
                client=original_payment.client,
                description=original_payment.description,
                template=original_payment.template,
                invoice_number=invoice_number,
                items=original_payment.items,
                amount=original_payment.amount,
                date_issued=timezone.now().date(),
                date_due=original_payment.date_due,
                date_paid=None,
                status='pending',
                type=original_payment.type,
            )

            # Копируем изображение
            if original_payment.image:
                file_name, file_ext = os.path.splitext(original_payment.image.name)
                new_file_name = f"{file_name}_copy{file_ext}"
                new_payment.image.save(
                    new_file_name,
                    ContentFile(original_payment.image.read()),
                    save=True
                )

            return Response(
                {"id": new_payment.id, "invoice_number": new_payment.invoice_number},
                status=status.HTTP_201_CREATED
            )

    except Exception as e:
        return Response(
            {"detail": f"Error creating duplicate: {str(e)}"},
            status=status.HTTP_500_INTERNAL_SERVER_ERROR
        )